// $Id: CComboBox.cpp,v 1.8 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CComboBox.hpp"
using Exponent::GUI::Controls::CComboBox;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CComboBox, CTextLabel)

//	===========================================================================
CComboBox::CComboBox(IControlRoot *root, const long uniqueId, const CRect &area, const CString &text, CMenu *menu, IActionListener *listener) 
		 : CTextLabel(root, uniqueId, area, text)
		 , m_menu(NULL)
		 , m_menuWindow(NULL)
		 , m_windowAttributes(NULL)
		 , m_useIcons(true)
		 , m_selectedIndex(0)
		 , m_state(false)
		 , m_downImage(NULL)
		 , m_selectedItem(NULL)
{
	EXPONENT_CLASS_CONSTRUCTION(CComboBox);

	if (menu == NULL)
	{
		throw CException("Menu is NULL", "CComboBox::CComboBox(IControlRoot *, const long, const CRect &, const CString &, CMenu *)");
	}

	this->setMenu(menu);
	this->controlIsMouseEnabled(true);
	this->controlIsDropFileEnabled(false);
	this->registerActionListener(listener);
}

//	===========================================================================
CComboBox::~CComboBox()
{
	EXPONENT_CLASS_DESTRUCTION(CComboBox);
	FORGET_COUNTED_OBJECT(m_menu);
	FORGET_COUNTED_OBJECT(m_downImage);
	FREE_POINTER(m_menuWindow);
	FREE_POINTER(m_windowAttributes);
	FORGET_COUNTED_OBJECT(m_selectedItem);
}

//	===========================================================================
void CComboBox::handleLeftButtonDown(CMouseEvent &event)
{
	if (m_menuWindow)
	{
		// Update our graphics
		m_state = true;
		this->update();

		// Open the menu window
		CPoint global;
		m_rootControl->getGlobalCoordinatesOfControl(this, global);
		global.offset(1, 0);
		m_menuWindow->openPopupWindow(global);
		#ifndef WIN32
			CWindowTools::setFocusWindow(m_menuWindow->getMutableWindowHandle());
		#endif
	}
}

//	===========================================================================
void CComboBox::handleMenuSelection(const CMenuEvent &event)
{
	this->setText((event.getMenuItem()) ? event.getMenuItem()->getText() : " ");

	// STore the menu item
	EXCHANGE_COUNTED_OBJECTS(m_selectedItem, event.getMenuItem());

	// Is this the selected index???
	m_selectedIndex = event.identify();

	// This should already have been called if they were just losing focus.
	// In any case its only a small overhead penalty (esp in GUI land) to still have it
	m_state = false;
	this->update();

	// Close the window...
	if (m_menuWindow)
	{
		m_menuWindow->closePopupWindow();
	}
	if (m_actionListener)
	{
		m_actionListener->handleActionEvent(CActionEvent(this));
	}
	#ifndef WIN32
		IWindow *parent = this->getRootControl()->getParentWindow();
		if (parent)
		{
			CWindowTools::setFocusWindow(parent->getMutableWindowHandle());
		}
	#endif
}

//	===========================================================================
void CComboBox::setMenu(CMenu *menu)
{
	EXCHANGE_COUNTED_OBJECTS(m_menu, menu);
}

//	===========================================================================
void CComboBox::setSelectedIndex(const long index)
{
	if (m_menu && index >= 0 && index < m_menu->getNumberOfMenuItems())
	{
		CMenuItem *item = m_menu->getMenuItem(index);
		if (item)
		{
			this->setText(item->getText());
		}
		
		EXCHANGE_COUNTED_OBJECTS(m_selectedItem, item);
		m_selectedIndex = index;
	}
}

//	===========================================================================
void CComboBox::setMenuWindowAttributes(CWindowAttributes *windowAttributes, const bool useIcons)
{
	if (m_windowAttributes == NULL)
	{
		m_windowAttributes = new CWindowAttributes();
	}

	*m_windowAttributes = *windowAttributes;
	m_useIcons			= useIcons;

	FREE_POINTER(m_menuWindow);
	m_menuWindow = new CMenuWindow(CRect(0, 0, m_area.getWidth(), 100), m_windowAttributes, m_menu, m_useIcons, this);
}

//	===========================================================================
void CComboBox::setFont(CFont *font)
{
	CTextLabel::setFont(font);
	if (m_menuWindow)
	{
		m_menuWindow->setFont(font);
	}
}

//	===========================================================================
void CComboBox::setMenuColours(const SMenuColours &colours)
{
	if (m_menuWindow)
	{
		m_menuWindow->setMenuColours(colours);
	}
}

//	===========================================================================
void CComboBox::setDownImage(IImage *downImage)
{
	EXCHANGE_COUNTED_OBJECTS(m_downImage, downImage);
}

//	===========================================================================
void CComboBox::drawControl(CGraphics &graphics)
{
	// First check if we can allow the standard handler to draw the disabled control
	if (!this->drawEnabledControl(graphics))
	{
		return;
	}

	// Draw the background
	if (!m_state)
	{
		this->drawPrimaryImage(graphics, m_doDefaultDrawing);
	}
	else if (m_downImage)
	{
		graphics.drawImage(m_downImage, m_normalisedArea, m_downImage->getNormalisedImageSize());
	}
	else
	{
		if (m_doDefaultDrawing)
		{
			graphics.drawRectangle(this->getNormalisedArea());
		}
	}

	// Draw the text string out
	graphics.setTextColour(m_textColour);
	graphics.drawText(m_text, CRect(1, 1, m_area.getWidth(), m_area.getHeight()), m_font, m_textFormat);
}

//	===========================================================================
void CComboBox::handleMenuClosing(const CMenuEvent &event) 
{
	m_menuWindow->setCurrentSelection(-1);
	m_state = false;
	this->update();
}

//	===========================================================================
CMenuItem *CComboBox::getMenuItem(const long index)
{
	return m_menu->getMenuItem(index);
}